package hivemind.example.article.hibernate;

import org.apache.hivemind.ServiceImplementationFactory;
import org.apache.hivemind.ServiceImplementationFactoryParameters;
import org.apache.hivemind.events.RegistryShutdownListener;
import org.apache.hivemind.service.ThreadCleanupListener;
import org.apache.hivemind.service.ThreadEventNotifier;
import org.apache.commons.logging.Log;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.hibernate.cfg.Configuration;

/**
 * @author James Carman
 * @version 1.0
 */
public class HibernateSessionFactory implements ServiceImplementationFactory, RegistryShutdownListener
{
//******************************************************************************
// Fields
//******************************************************************************

    private SessionFactory sessionFactory;
    private ThreadEventNotifier notifier;
    private boolean updateSchema = true;
    private Log log;

//******************************************************************************
// Getters/Setters
//******************************************************************************

    public void setNotifier( ThreadEventNotifier notifier )
    {
        this.notifier = notifier;
    }

    public void setLog( Log log )
    {
        this.log = log;
    }

    public void setUpdateSchema( boolean updateSchema )
    {
        this.updateSchema = updateSchema;
    }
//******************************************************************************
// Interface RegistryShutdownListener
//******************************************************************************

    public void registryDidShutdown()
    {
        log.debug( "Closing Hibernate SessionFactory..." );
        sessionFactory.close();
    }

//******************************************************************************
// Interface ServiceImplementationFactory
//******************************************************************************

    public Object createCoreServiceImplementation( ServiceImplementationFactoryParameters params )
    {
        log.debug( "Creating Hibernate Session..." );
        final Session session = sessionFactory.openSession();
        notifier.addThreadCleanupListener( new SessionCloser( session ) );
        return session;
    }

//******************************************************************************
// Other Methods
//******************************************************************************

    public void init()
    {
        log.debug( "Initializing Hibernate SessionFactory..." );
        final Configuration config = new Configuration();
        config.configure();
        if( updateSchema )
        {
            log.debug( "Updating database schema..." );
            new SchemaUpdate( config ).execute( true, true );
        }
        sessionFactory = config.buildSessionFactory();
    }

//******************************************************************************
// Inner Classes
//******************************************************************************

    private class SessionCloser implements ThreadCleanupListener
    {
        private final Session session;

        public SessionCloser( Session session )
        {
            this.session = session;
        }

        public void threadDidCleanup()
        {
            log.debug( "Closing Hibernate Session..." );
            session.close();
            notifier.removeThreadCleanupListener( this );
        }
    }
}

