package hivemind.example.article.interceptor;

import hivemind.example.article.service.TransactionService;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.LogFactory;
import org.jmock.Mock;
import org.jmock.MockObjectTestCase;

/**
 * @author James Carman
 * @version 1.0
 */
public class TestTransactionInterceptor extends MockObjectTestCase
{
//******************************************************************************
// Fields
//******************************************************************************

    Mock transactionServiceMock;
    TransactionInterceptor interceptor;

//******************************************************************************
// Other Methods
//******************************************************************************

    protected void setUp() throws Exception
    {
        transactionServiceMock = mock( TransactionService.class );
        interceptor = new TransactionInterceptor();
        interceptor.setTransactionService( ( TransactionService ) transactionServiceMock.proxy() );
        interceptor.setLog( LogFactory.getLog( TransactionService.class ) );
    }

    protected void tearDown() throws Exception
    {
        transactionServiceMock = null;
    }

    public void testWithTransaction() throws Throwable
    {
        expectsActive( true );
        assertProceeds();
    }

    public void testWithRollbackOnly() throws Throwable
    {
        expectsActive( false );
        expectsBegin();
        expectsIsRollbackOnly( true );
        expectsRollback();
        assertProceeds();
    }

    public void testWithApplicationException() throws Throwable
    {
        expectsActive( false );
        expectsBegin();
        expectsIsRollbackOnly( false );
        expectsCommit();
        assertProceeds( new Exception() );
    }

    public void testWithRuntimeException() throws Throwable
    {
        expectsActive( false );
        expectsBegin();
        expectsSetRollbackOnly();
        expectsIsRollbackOnly( true );
        expectsRollback();
        assertProceeds( new RuntimeException() );
    }

    private void assertProceeds( Exception e ) throws Throwable
    {
        final Mock methodInvocationMock = mock( MethodInvocation.class );
        methodInvocationMock.expects( once() ).method( "proceed" ).withNoArguments().will( throwException( e ) );
        try
        {
            interceptor.invoke( ( MethodInvocation ) methodInvocationMock.proxy() );
            fail();
        }
        catch( Exception thrown )
        {
            assertEquals( e, thrown );
        }
    }

    private void expectsSetRollbackOnly()
    {
        transactionServiceMock.expects( once() ).method( "setRollbackOnly" ).withNoArguments();
    }

    private void expectsActive( boolean returnValue )
    {
        transactionServiceMock.expects( once() ).method( "isActive" ).withNoArguments().will( returnValue( returnValue ) );
    }

    private void assertProceeds()
    {
        final Mock methodInvocationMock = mock( MethodInvocation.class );
        methodInvocationMock.expects( once() ).method( "proceed" ).withNoArguments().will( returnValue( null ) );
        try
        {
            interceptor.invoke( ( MethodInvocation ) methodInvocationMock.proxy() );
        }
        catch( Throwable t )
        {
            fail();
        }
    }

    public void testWithoutTransaction() throws Throwable
    {
        expectsActive( false );
        expectsBegin();
        expectsIsRollbackOnly( false );
        expectsCommit();
        assertProceeds();
    }

    private void expectsBegin()
    {
        transactionServiceMock.expects( once() ).method( "begin" ).withNoArguments();
    }

    private void expectsIsRollbackOnly( boolean returnValue )
    {
        transactionServiceMock.expects( once() ).method( "isRollbackOnly" ).withNoArguments().will( returnValue( returnValue ) );
    }

    private void expectsCommit()
    {
        transactionServiceMock.expects( once() ).method( "commit" ).withNoArguments();
    }

    private void expectsRollback()
    {
        transactionServiceMock.expects( once() ).method( "rollback" ).withNoArguments();
    }
}