package EJBObserverPattern;

import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import javax.ejb.EJBException;

/**
 * A class to be subclassed by any entity which may be observed
 * by instances of EJBObserver. The behavior of this object is
 * analagous to that of java.util.Observable.
 *
 * @see     java.util.Observable
 */
public class EJBObservable implements java.io.Serializable
{
    /**
     * The collection of observers. Using a Set will guarantee
     * no duplicate observers.
     */
    protected Set mObservers = new HashSet();

    /**
     * The <i>changed</i> indicator. This indicator must be set to true
     * via a setChanged(true) method call in order for notifyObservers
     * to execute the notification process.
     */
    protected boolean mChanged = false;

    /**
     * Add an observer to the list of observers for this instance.
     * @param   observer    an EJBObserver
     */
    public void addObserver (EJBObserver observer)
    {
        mObservers.add (observer);
    }

    /**
     * Delete an observer from the list of observer for this instance.
     * @param   observer    an EJBObserver
     */
    public void deleteObserver (EJBObserver observer)
    {
        mObservers.remove (observer);
    }

    /**
     * Delete all observers from this instance.
     */
    public void deleteObservers ()
    {
        mObservers.clear();
    }

    /**
     * Return the number of observers registered for this instance.
     * @return      an observer count
     */
    public int countObservers ()
    {
        return mObservers.size();
    }

    /**
     * Return an indicator as to whether this instance has changed.
     * @return      a changed indicator
     */
    public boolean hasChanged ()
    {
        return mChanged;
    }

    /**
     * Clear the changed indicator. This is identical to calling
     * setChanged (false).
     */
    public void clearChanged ()
    {
        this.setChanged (false);
    }

    /**
     * Set the changed indicator to the given value.
     * @param       changed     true or false
     */
    protected void setChanged (boolean changed)
    {
        mChanged = changed;
    }

    /**
     * An overloaded form of notifyObservers that takes no argument. The semantics
     * are identical to calling notifyObservers (..., null).
     * @exception   RemoteException     thrown by a remote implementation of EJBObserver
     * @exception   EJBException        thrown by a remote implementation of EJBObserver
     */
    public void notifyObservers () throws RemoteException, EJBException
    {
        this.notifyObservers (null);
    }

    /**
     * If this object has changed as indicated by the hasChanged method,
     * notify all observers and call clearChanged to reset the hasChanged indicator.
     * @param       arg   any object
     * @exception   RemoteException     thrown by a remote implementation of EJBObserver
     * @exception   EJBException        thrown by a remote implementation of EJBObserver
     */
    public void notifyObservers (Object arg) throws RemoteException, EJBException
    {
        if (this.hasChanged())
        {
            for (Iterator i = mObservers.iterator(); i.hasNext(); )
            {
                EJBObserver obs = (EJBObserver) i.next();

                obs.update (this, arg);

            }

            this.clearChanged();

        }
    }
}