package EJBObserverPattern;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.rmi.RemoteException;
import java.util.Properties;
import java.util.Set;
import java.util.Iterator;
import java.util.HashSet;
import javax.ejb.EJBException;
import javax.ejb.EJBHome;
import javax.rmi.PortableRemoteObject;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

/**
 * A proxy for a remote implementation of EJBObserver.
 * <p>This class makes the use of EJBObserver more efficient by
 * delaying the lookup of a home interface and the creation of
 * a remote interface until such time as they are needed.
 * The mechanics of looking up the home and remote interfaces
 * and invoking the remote observer are delegated to the update method of
 * this class.
 */
public class EJBObserverProxy implements java.io.Serializable, EJBObserver
{
    /**
     * The initial context properties for the EJBObserver implementation that
     * is being proxied.
     */
    private Properties mContextValues = null;

    /**
     * The JNDI name of the EJBObserver implementation that is being proxied.
     */
    private String mJNDIName = null;

    /**
     * Creates an instance of this class to proxy an EJBObserver specified
     * by the given arguments.
     *
     * @param   initialContextFactory   the initial context factory to use when
     * creating an InitialContext to use in obtaining the home interface of the
     * EJBObserver this is being proxied
     * @param   providerUrl             the provider url to use when creating
     * an InitialContext to use in obtaining the home interface of the EJBObserver
     * that is being proxied
     * @param   jndiName        the JNDI name of the EJBObserver that is being proxied
     */
    public EJBObserverProxy (
        String initialContextFactory,
        String providerUrl,
        String jndiName)
    {
        Properties p = new Properties();
        p.put(Context.INITIAL_CONTEXT_FACTORY,
            initialContextFactory);

        p.put(Context.PROVIDER_URL,
            providerUrl);

        init (p, jndiName);
    }

    /**
     * Creates an instance of this class to proxy an EJBObserver specified
     * by the given arguments.
     *
     * @param   contextValues   a Properties instance that will be used to construct
     * an InitialContext object when the time comes to invoke the EJBObserver that
     * is being proxied
     * @param   jndiName        the JNDI name of the EJBObserver that is being proxied
     */
    public EJBObserverProxy (
        Properties contextValues,
        String jndiName )
    {
        init (contextValues, jndiName);
    }

    /**
     * Initialize this instance with the given values. Called by the public
     * constructors of this class.
     *
     * @param   contextValues   a Properties instance that will be used to construct
     * an InitialContext object when the time comes to invoke the EJBObserver that
     * is being proxied
     * @param   jndiName        the JNDI name of the EJBObserver that is being proxied
     */
    protected void init (
        Properties contextValues,
        String jndiName )
    {
        mContextValues = contextValues;
        mJNDIName = jndiName;
    }

    /**
     * Returns the JNDI name of the EJBObserver being proxied.
     * @return  the JNDI name of the EJBObserver being proxied.
     */
    public String getName ()
    {
        return mJNDIName;
    }

    /**
     * Returns the Properties instance that is used to construct an InitialContext
     * to lookup the home interface of the EJBObserver being proxied
     * @return  the Properties instance that is used to construct an InitialContext
     * to lookup the home interface of the EJBObserver being proxied
     */
    public Properties getContextValues ()
    {
        return mContextValues;
    }

    /**
     * Combines the member variables of this object to calculate a hash code.
     * This will make instances of this class well-behaved members of a Collection.
     * @return  a hash code
     */
    public int hashCode ()
    {
        return (mContextValues.toString() + mJNDIName).hashCode();
    }

    /**
     * Returns an indicator as to whether the given object is functionally equal to this
     * object. This method will return true if the given object is an instance of
     * EJBObserverProxy and the mContextValues and mJNDIName attributes
     * are equal to the respective attributes of this class. Equality is tested using
     * the equals method of the corresponding classes of the given attributes.
     *
     * @param       o
     * @return      true or false whether the given object is functionally
     * equal to this object
     */
    public boolean equals (Object o)
    {
        EJBObserverProxy loc = (EJBObserverProxy) o;
        return (
            (mContextValues.equals (loc.getContextValues())) &&
            (mJNDIName.equals (loc.getName()))
            );
    }

    /**
     * This method is a proxy for the update method of the encapsulated
     * EJBObserver implementation. When this method is called the EJBObserver
     * is instantiated and invoked using the arguments specified when this object
     * was constructed.
     *
     * @param       observable
     * @param       arg
     * @exception   RemoteException
     * @exception   EJBException
     */
    public void update (EJBObservable object, Object arg)
    throws RemoteException, EJBException
    {
        try
        {
            String beanHomeName = this.getName();

            Properties p = this.getContextValues();

            InitialContext namingContext = new InitialContext(p);

            /*  Get a reference to the named object and narrow it to EJBHome
                to make sure it is a home interface.
            */

            EJBHome obsHome = (EJBHome) PortableRemoteObject.narrow (
                namingContext.lookup (beanHomeName), EJBHome.class );

            /*  We can't directly cast this reference to a home interface because each
                EJB implementation will have its own home interface in some arbitrary
                class hierarchy. EJBHome doesn't define a create() method
                so we can't invoke it via polymorphism.

                Let's use reflection to find and invoke the create() method.
                Look for a method named 'create' which takes no parameters.

            */

            Method createMethod = obsHome.getClass().getDeclaredMethod (
                "create", new Class[0] );

            /*  Invoke the 'create' method of the home interface via our reflected Method.
            */

            Object remoteObj = createMethod.invoke (obsHome, new Object[0]);

            /*  Narrow the resulting reference to the EJBObserver
                interface, which each remote observer is obligated to implement.
            */

            EJBObserver obs = (EJBObserver) PortableRemoteObject.narrow (
                remoteObj, EJBObserver.class);

            /*  Update this observer with an EJBObservable instance and some argument.
                Foobar is an instance of EJBObservable.
            */

            obs.update (object, arg);
        }
        catch (NoSuchMethodException e)
        {
            throw new EJBException (e);
        }
        catch (NamingException e)
        {
            throw new EJBException (e);
        }
        catch (InvocationTargetException e)
        {
            throw new EJBException (e);
        }
        catch (IllegalAccessException e)
        {
            throw new RuntimeException (e.toString());
        }
    }
}

