/*
Copyright 2004 Matt Butcher

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package tv.aleph_null.opencms.velocity;

import java.util.Properties;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import com.opencms.boot.I_CmsLogChannels;
import com.opencms.core.A_OpenCms;
import com.opencms.core.CmsException;
import com.opencms.core.I_CmsRequest;
import com.opencms.flex.jsp.CmsJspNavBuilder;
import com.opencms.file.CmsFile;
import com.opencms.file.CmsGroup;
import com.opencms.file.CmsObject;
import com.opencms.file.CmsUser;
import com.opencms.template.CmsXmlControlFile;
import com.opencms.template.CmsXmlTemplateFile;
import com.opencms.util.LinkSubstitution;
import com.opencms.util.Utils;

/**
 * This class provides tools that will be useful in velocity templates.
 * This class should be instantiated and then placed in the Context for
 * a template.
 */
public class CmsVelocityTools {

	CmsObject cms;
	String errMsg = "";
	boolean isErr = false;
	VelocityContext ctx;

	/**
	 * Main constructor.
	 * Most methods that this class contains will use a CmsObject for one thing
	 * or another, so this object must have a CmsObject.
	 * @param cms CmsObject for manipulating the CMS.
	 */
	public CmsVelocityTools( CmsObject cms, VelocityContext ctx ) {
		this.cms = cms;
		this.ctx = ctx;
		return;
	}

	/**
	 * Rewrites links. This will take a reference to a resource and create a URI
	 * for linking to the resource. This should work identically to the JSP
	 * &lt;cms:link/&rt; directive.
	 * @param linkMe String that will be rewritten as a URI.
	 * @return String value suitable for inclusion in a hyperlink href.
	 * @see com.opencms.util.LinkSubstitution#getLinkSubstitution(CmsObject,String)
	 * @see com.opencms.util.CmsJspTagLink#linkTagAction(String,CmsFlexRequest)
	 */
	public String link( String linkMe ) {
		if(linkMe == null || "".equals(linkMe)) return "";
		return LinkSubstitution.getLinkSubstitution(this.cms, linkMe);
		/*
		String link = null;
		if(linkMe.indexOf(':') >= 0) { 
			// Looks like URL (http://, ftp://, https://)?
			link = LinkSubstitution.getLinkSubstitution(this.cms, linkMe);
		} else {
			// We cant get a handle on a CmsFlexRequest b/c we are bypassing JSP
			//link = LinkSubstitution.getLinkSubstitution(this.cms, CmsFlexRequest.toAbsolute(linkMe));
			I_CmsRequest req = this.cms.getRequestContext().getRequest();
			StringBuffer sb = new StringBuffer(256); // Max len of GET
			sb.append(req.getScheme());
			sb.append("://";
			sb.append(req.getServerName());
			sb.append(":");
			sb.append(req.getServerPort());
			sb.append(getRequestUri());
			try {
				url = new java.net.URL(new java.net.URL(sb.toString(), linkMe);
				// FIXME: Stopped at line 292 of cache.CmsFlexRequest
			} catch(java.net.MalformedURLException murle) {
				return LinkSubstitution.getLinkSubstitution(this.cms, linkMe);
			}
		}
		*/
	}
	
	/**
	 * Get navigation for this page.
	 * Gets a com.opencms.flex.jsp.CmsJspNavBuilder instance. Note that each
	 * time you call this, a new navigation object will be created.
	 * @see com.opencms.flex.jsp.CmsJspNavBuilder
	 */
	public CmsJspNavBuilder getNavigation() {
		CmsJspNavBuilder nav = null;
		try {
			return new CmsJspNavBuilder(this.cms);
		} catch (Throwable t) {
			this.setError("Failed to get navigation. ", t);
		}
		return null;
	}

	/**
	 * Gets user information. This returns a CmsUser object.
	 * Example Use of a CmsUser: 
	 * <xmp>
	 * #set($user, cmstools.User)
	 * Current user is $user.Firstname $user.Lastname
	 * </xmp>
	 * @return CmsUser object
	 * @see com.opencms.file.CmsUser
	 */
	public CmsUser getUser() {
		return cms.getRequestContext().currentUser();
	}

	/**
	 * Gets group. This returns a CmsGroup object.
	 * Example Use of a CmsGroup: 
	 * <xmp>
	 * #set($group, cmstools.Group)
	 * Current group is $group.Name
	 * </xmp>
	 * @return CmsUser object
	 * @see com.opencms.file.CmsUser
	 */
	public CmsGroup getGroup() {
		return cms.getRequestContext().currentGroup();
	}

	/**
	 * Set an error message for the template to get.
	 * Basically, this can be used for providing simple info to the template for
	 * non-fatal errors (like failing to generate navigation). I'm fairly new
	 * to Velocity. Let me know if there is an existing pattern for this.
	 * <p>using this method will result in isError being set to true.</p>
	 * @param msg String containing an error message
	 */
	private void setError(String msg) {
		this.isErr = true;
		this.errMsg += msg;
		return;
	}

	/**
	 * Set an error message for the template to get.
	 * Basically, this can be used for providing simple info to the template for
	 * non-fatal errors (like failing to generate navigation). I'm fairly new
	 * to Velocity. Let me know if there is an existing pattern for this.
	 * <p>using this method will result in isError being set to true.</p>
	 * <p>This will also log the throwwable (usually an exception) with
	 * the OpenCms logger (Using the C_MODULE_INFO channel).</p>
	 * @param msg String containing an error message. This error should be 
	 * printable by the template designer, so don't be dumb and put security-
	 * sensitive info in the msg.
	 */
	private void setError(String msg, Throwable t) {
		if(I_CmsLogChannels.C_LOGGING 
				&& A_OpenCms.isLogging(I_CmsLogChannels.C_MODULE_INFO)) {
			A_OpenCms.log( I_CmsLogChannels.C_MODULE_INFO, 
						   Utils.getStackTrace(t));
		}
		this.setError(msg);
		return;
	}

	/**
	 * Returns true if there is an error. False, otherwise.
	 * @return true if an error occured.
	 */
	public boolean isError() {
		return this.isErr;
	}

	/**
	 * Clears all error messages and resets isErr to false.
	 * This is public so that template developers can handle errors a little
	 * more easily.
	 */
	public void clearError() {
		this.isErr = false;
		this.errMsg = "";
	}

	/**
	 * Runs $content through the velocity engine. Using this, you can run
	 * VTL code that is stored within content bodies. That means that you can
	 * write velocity code in the WYSIWYG editor and then have velocity 
	 * interpret the resulting text as a template. That also means that your
	 * editors can write code -- buyer beware!
	 * <p>Details:
	 * <ul>
	 * <li>Uses the same VelocityContext as was present in the original 
	 * request.</li>
	 * <li>Uses the value of "content" in the VelocityContext</li>
	 * <li>Does not attempt to resolve HTML entites (e.g. &amp;amp;)</li>
	 * </ul>
	 * @return a String containing an interpreted version of $content.
	 */
	public String parseContent() {
		try {
			java.io.StringWriter out = new java.io.StringWriter();
			Velocity.evaluate( this.ctx, 
							   out, 
							   "VELOCITY", 
							   (String)this.ctx.get("content"));
			return out.toString();
		} catch (Exception e) {
			this.setError("Failed to parse content", e);
		}
		return "";
	}

	/**
	 * Includes the contents of the specified file.
	 * The file name should be the ABSOLUTE path to the file in the CMS (not
	 * as a URI). E.g. /myfiles/foo.txt, not /opencms/opencms/myfiles/foo.txt.
	 * <p>It will return the full contents of the file as a String object. 
	 * Supported types are page and plain. Binary and image files cannot be
	 * fetched with this method. And JSP files are right out.<p>
	 * <p>Example:</p>
	 * <xmp>
	 * $cmstools.include("/mydir/myfile.txt")
	 * </xmp>
	 * <p><b>Velocity Templates Are Not Parsed By Include()!</b></p>
	 * <p>For better or worse, include() does not place restrictions on which 
	 * files in the CMS may be included, though they must be in the same project
	 * .</p>
	 * @param fileName Name of the file.
	 * @return full contents of the file in a String.
	 * @see #getAlternateBody(String,String)
	 */
	public String include(String fileName) {
		String content = "";
		CmsFile resFile;
		try {
			resFile = this.cms.readFile( fileName );
			int fileType = resFile.getType();

			if(fileType == this.cms.getResourceType("page").getResourceType()) {
				// Must get the content from the body file:
				CmsXmlControlFile controlCode = 
					new CmsXmlControlFile(this.cms, resFile);

				String bodyFileName = controlCode.getElementTemplate("body");
				CmsXmlTemplateFile bodyFile = 
					new CmsXmlTemplateFile( this.cms, bodyFileName );

				content = bodyFile.getTemplateContent( null, null, null );

			} else if( fileType == 
					this.cms.getResourceType("plain").getResourceType()) {
				//Just get the content...
				content = new String(resFile.getContents());
			} else {
				this.setError("Can't get files of that type.");
			}
		} catch(CmsException cmse) {
			this.setError("Cannot read file", cmse);
			return content;
		}
		return content;
	}

	/**
	 * Sends the contents of fileName into the Velocity parser and returns the
	 * results. <b>USE WITH CARE</b>.
	 * The file name should be the ABSOLUTE path to the file in the CMS (not
	 * as a URI). E.g. /myfiles/foo.txt, not /opencms/opencms/myfiles/foo.txt.
	 * <p>For better or worse, parse() does not place restrictions on where
	 * in the CMS files are loaded from. Files must be type plain, though.</p>
	 * <p>The Context used for this template is inherited from the parent
	 * page. In the future, this may change slightly. </p>
	 * @param fileName name of the file to parse.
	 * @return full contents of the evaluated template.
	 * @see #include(String)
	 */
	public String parse(String fileName) {
		String content;
		CmsFile resFile;
		try {
			resFile = this.cms.readFile( fileName );
			int fileType = resFile.getType();
			if( fileType == 
					this.cms.getResourceType("plain").getResourceType()) {
				//Just get the content...
				content = new String(resFile.getContents());
			} else {
				this.setError("Can't get files of that type.");
				return "";
			}
		} catch (CmsException cmse) {
			this.setError("Could not open file for parsing ",cmse);
			return "";
		}
		if(content != null && !"".equals(content)) {
			try {		
				java.io.StringWriter out = new java.io.StringWriter();
				// No need to re-init() a Singleton.
				//Properties p = new Properties();
				//ClassLoader cl = Thread.currentThread().getContextClassLoader();
				//p.load(cl.getResourceAsStream(CmsVelocityTemplate.VELOCITY_PROPS_FILE));
				//Velocity.init(p);
				Velocity.evaluate( this.ctx, out, "VELOCITY", content);
				return out.toString();
			} catch (Exception e) {
				this.setError("Failed to include(parse) file.", e);
			}
		}
		return "";
	}

	/**
	 * Includes the specified body of the specified file.
	 * The file name should be the ABSOLUTE path to the file in the CMS (not
	 * as a URI). E.g. /myfiles/foo.txt, not /opencms/opencms/myfiles/foo.txt.
	 * <p>It will return the full contents of the file body as a String object. 
	 * Only Page types are supported.</p>
	 * @param fileName name of the file
	 * @param bodyName name of the body in a multi-body Page
	 * @return full contents of the body in a String
	 * @see #include(String)
	 */
	public String getAlternateBody(String fileName, String bodyName) {
		String content = "";
		try { 
			CmsFile resFile = this.cms.readFile( fileName );
			int fileType = resFile.getType();

			if(fileType == this.cms.getResourceType("page").getResourceType()) {
				// Must get the content from the body file:
				CmsXmlControlFile controlCode = 
					new CmsXmlControlFile(this.cms, resFile);

				String bodyFileName = controlCode.getElementTemplate("body");
				CmsXmlTemplateFile bodyFile = 
					new CmsXmlTemplateFile( this.cms, bodyFileName );

				content = bodyFile.getTemplateContent( null, null, bodyName );
			} else {
				this.setError("Only Pages support alternate bodies.");
			}
		} catch (CmsException cmse) {
			this.setError("Could not get alternate body for file.", cmse);
		} 
		return content;
	}

	/**
	 * Returns an error string set by setError(). If no error has been set,
	 * then this will return an empty string. Note: Errors are concatenated, 
	 * so multiple error messages may be present. Use isError() to see if an
	 * error has occured.
	 * @return an error message String.
	 */
	public String getError() {
		return this.errMsg;
	}
}
