/*
Copyright 2004 Matt Butcher

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package tv.aleph_null.opencms.velocity;

import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import java.io.StringWriter; //just for output.

import org.apache.velocity.app.Velocity;
import org.apache.velocity.VelocityContext;

import com.opencms.core.A_OpenCms;
import com.opencms.core.CmsException;
import com.opencms.file.CmsRequestContext;
import com.opencms.file.CmsFile;
import com.opencms.file.CmsObject;
import com.opencms.flex.CmsJspTemplate;
import com.opencms.template.CmsXmlControlFile;
import com.opencms.template.CmsXmlTemplateFile;

/**
 * This class provides OpenCms with a Velocity template handler.
 * In OpenCms 5.0.x, an XML Template is called first for every request. Usually,
 * it then loads a JSP template. This class is designed to load Velocity 
 * instead of the CmsJspTemplate evaluator.
 *
 * It extends com.opencms.flex.CmsJspTemplate because caching of the two are
 * handled just about identically.
 *
 * @author Matt Butcher (mmbutcher@aleph-null.tv)
 * @version 0.1
 */
public class CmsVelocityTemplate extends CmsJspTemplate {

	/**
	 *	Default: tv/aleph_null/opencms/velocity/velocity.properties
	 */
	public static String VELOCITY_PROPS_FILE = //"velocity.properties";
		"tv/aleph_null/opencms/velocity/velocity.properties";

	public CmsVelocityTemplate() {
		// NOOP
	}

	/**
	 * Handles a request using Velocity templates.
	 * This method takes a velocity template and uses it to format a response.
	 * Currently, it works for Page types. 
	 * <p><b>Note to velocity designers:</b> #include and #parse will try to 
	 * load using classpaths, which will not find resources in the CMS. 
	 * (Alternatives: 
	 * @{link CmsVelocityTools#include(String)} and 
	 * @{link CmsVelocityTools#parse(String)})</p>
	 * <p>The following items get placed in the Velocity context object:
	 * <ul>
	 * <li>$properties - contains the document propertes. e.g.
	 * <code>$properties.get("Title")</code></li>
	 * <li>$content - content of a Page (HTML fragment).</li>
	 * <li>$request - request object (com.opencms.core.I_CmsRequest)</li>
	 * <li>$cms - instace of com.opencms.file.CmsObject</li>
	 * <li>$cmstools - helper tools from CmsVelocityTools. e.g. 
	 * <code>$cmstools.link(file)</code></li>
	 * <li>$session - The CmsSession object.</li>
	 * </ul>
	 *
	 * @param cms CmsObject for CMS access.
	 * @param velocityTemplate Velocity template file name.
	 * @param elementName unused
	 * @param parameters unused
	 * @return Template content
	 * @throws CmsException if something goes wrong
	 * @see com.opencms.file.CmsObject
	 * @see tv.aleph_null.opencms.velocity.CmsVelocityTools
	 */
	public byte[] getContent( CmsObject cms, 
							  String velocityTemplate,
							  String elementName,
							  Hashtable parameters )
							  throws CmsException
	{
		String content = null;
		byte[] res = null;
		try {
			//First, get resource that we are going to put in template:
			CmsRequestContext requestCtx = cms.getRequestContext();
			String fileName = requestCtx.getUri();

			// Next, get file properties Map:
			Map props = cms.readProperties(fileName);

			// Now, get file content:
			CmsFile resFile = cms.readFile( fileName );
			int fileType = resFile.getType();
			/*
			 * We will only handle files of two types: Page and Plain. JSP,
			 * binary and other types should not come through velocity.
			 */	
			if(fileType == cms.getResourceType("page").getResourceType()) {
				// Must get the content from the body file:
				CmsXmlControlFile controlCode = 
					new CmsXmlControlFile(cms, resFile);

				String bodyFileName = controlCode.getElementTemplate("body");
				CmsXmlTemplateFile bodyFile = 
					new CmsXmlTemplateFile( cms, bodyFileName );

				content = bodyFile.getTemplateContent( null, null, null );

			} else if( fileType == 
					cms.getResourceType("plain").getResourceType()) {
				//Just get the content...
				content = new String(resFile.getContents());
			} else {
				throw new CmsException( "Unsuported file type. Only Plain and "
					+ "Page documents can be handled by CmsVelocityTemplate.");
			}

			// We also need to get the content of the template:
			CmsFile templateFile = cms.readFile(velocityTemplate);
			String templateContents = new String(templateFile.getContents());


			// At this point, we can start dumping things into the velocity
			// context:
			VelocityContext ctx = new VelocityContext();
			CmsVelocityTools tools = new CmsVelocityTools(cms, ctx);
			ctx.put("properties", props);
			ctx.put("content", content);
			ctx.put("request", requestCtx.getRequest());
			//ctx.put("response", requestCtx.getResponse());
			ctx.put("session", requestCtx.getSession(false));
			ctx.put("cms", cms);
			ctx.put("cmstools", tools);

			// Okay... all we need to do is get the output writer and then
			// run Velocity:
			StringWriter out = new StringWriter();

			Properties p = new Properties();
			ClassLoader cl = Thread.currentThread().getContextClassLoader();
			try {
				p.load(cl.getResourceAsStream(VELOCITY_PROPS_FILE));
			} catch (NullPointerException npe) {
				throw new CmsException("Could not find " + VELOCITY_PROPS_FILE, 
					npe);
			}
			Velocity.init(p);
			Velocity.evaluate( ctx, out, "VELOCITY", templateContents);

			res =  out.toString().getBytes();
		} catch( Exception e ) {
			throw new CmsException(
				"[CmsVelocityTemplate] Error with velocity file: "
				+ velocityTemplate + "\n"
				+ e.toString(), 
				e);
		}
		return res;
	}
}
