package example;

/**
 * Builds the example.
 *
 * @author <a href="mailto:cristina@acm.org">Cristina Belderrain</a>
 */
public class ExampleBuilder {

    private int nbEvaluators;

    private Evaluator[] evaluators;

    private Thread[] threads;

    public ExampleBuilder() {
        nbEvaluators = 3;
        evaluators = new Evaluator[nbEvaluators];
        threads = new Thread[nbEvaluators];
    }

    public void runExample() {
        // Setup
        // Typical user profiles
        UserProfile richGuy = new UserProfile("Bob", 80000, true);
        UserProfile averageGuy = new UserProfile("Dave", 40000, true);
        UserProfile poorGuy = new UserProfile("Jack", 20000, false);

        // Choose a profile at random...
        UserProfile profile = null;
        double n = (double) (Math.random() * 3);
        if (n >= 0 & n < 1) {
            profile = poorGuy;
        }
        else {
            if (n >= 1 & n < 2) {
                profile = averageGuy;
            }
            else profile = richGuy;
        }

        // Immutable state
        ImmutableState immutable = new ImmutableState(100000, 10000, 100, 25, profile);
        // Initial problem state
        ProblemState sampleState = new ProblemState(immutable);

        // Evaluators
        evaluators[0] = new IncomeEvaluator(sampleState);
        evaluators[1] = new RiskEvaluator(sampleState);
        evaluators[2] = new CreditEvaluator(sampleState);

        System.out.println(".: Encapsulated Business Rule Pattern Example :.");
        // Show initial state
        sampleState.immutState.profile.display();
        // Rule tracing
        System.out.println(" \nRules were applied as follows:");

        // Run the example
        // Start threads
        for (int i = 0; i < nbEvaluators; i++) {
            threads[i] = new Thread(evaluators[i]);
            threads[i].start();
        }
        // Wait for threads to join
        for (int i = 0; i < nbEvaluators; i++) {
            try {
                threads[i].join();
            }
            catch (InterruptedException ex) {
            }
        }

        // Show final state
        sampleState.display();
    }
}
