package examples.ejb20.basic.beanManagedDependentValueObjects;

import java.util.*;


/**
 * A set for storing dependent value objects and tracking their state
 * (Created/original/Updated/Deleted).
 *
 * @author: Floyd Marinescu, Senior Architect, The Middleware Company.
 * The Middleware Company is an Enterprise Java Training company. Visit our
 * website at: www.middleware-company.com.
 * Copyright 2001 The Middleware Company.
 *
 * This class is provided as an example only. As cool as it is, the author, 
 * The Middleware Company and TheServerSide.com take no responsibility for the
 * use or mis-use of this class.
 */
public class DependentValueSet implements Set {

	//stores references to dependent objects that existed in the DB when the list was
	//first loaded
	protected HashSet original = new HashSet();

	//stores original objects, new objects, and updated objects
	//this list reflects all the depenent objects that exist
	//at any point in time.
	protected HashSet workingList = new HashSet();

	//stores original dependent objects that were deleted
	protected HashSet deleted = new HashSet();

	//stores old dependent objects that have been updated
	protected HashMap updatedMap = new HashMap();
	
	/**
	 * DependentValueSet constructor, nothing going on here.
	 */
	public DependentValueSet() {

	}
	
	public boolean add(Object o) {

		//add object to our working list (these will be inserted in ejbStore()
		return this.workingList.add(o);

	}
	
	public boolean addAll(Collection c) {

		//add object to our working list (these will be inserted in ejbStore()
		return this.workingList.addAll(c);
		
	}
	
	
	/**
	 * Takes a dependent object loaded straight from the database.
	 * To be used only when this list is first loaded.
	 */
	public void addFromDatabase(Object o)
	{
		this.original.add(o);
		this.workingList.add(o);
		
	}
	
	
	public void clear() 
	{
		//delete all original dependent objects from the DB
		this.deleted.addAll( this.original );
		this.original.clear();

		//delete any new/modified objects from memory (since these aren't in the DB yet
		this.workingList.clear();
	}
	
	
	public boolean contains(Object o) {
		return this.workingList.contains(o);
	}
	
	
	public boolean containsAll(Collection c) {
		return this.workingList.containsAll(c);
	}
	
	
	/**
	 * Copies workinglist into original list, after database has been sychronized
	 *
	 * This method should only be called after allions have succeeded in ejbStore().
	 * It is necessary incase the bean is being cached in between transactions, or if 
	 * ejbStore is being called within the transaction (for synchronization or passivation).
	 */
	public void databaseHasBeenUpdated() 
	{
		//add all objects to original list, since they are now in the DB
		this.original.clear();
		this.original.addAll(this.workingList );

		//clear all other lists
		this.deleted.clear();
		this.updatedMap.clear();

	}
	
	
	/**
	 * To be called in ejbStore, all objects returned here
	 * need to be deleted from the underlying database.
	 */
	public Collection getObjectsToDelete() {
		return Collections.unmodifiableSet(this.deleted);
	}
	
	
	/**
	 * To be called in ejbStore(). The returned collection
	 * contains all objects that were created or updated
	 * in the current transaction. These objects need to be
	 * inserted into the database. 
	 */
	public Collection getObjectsToInsert() {


		Collection tempCollection = (Collection) this.workingList.clone();

		//get all new or updated objects
		//to do this, remove any original (already in the DB)
		//objects still in the working list
		tempCollection.removeAll(this.original);
		tempCollection.removeAll(this.updatedMap.keySet());

		//we don't really need to do this, but just to be consistent
		return Collections.unmodifiableCollection(tempCollection);
			
	}
	
	
	/**
	 * To be called in ejbStore, returns a map whose keys are
	 * the new updated dependent objects and whose values are the original
	 * dependent objects still in the database.
	 */
	public Map getObjectsToUpdate() {
		
		return Collections.unmodifiableMap(this.updatedMap);
	}
	
	
	public boolean isEmpty() {
		return this.workingList.isEmpty();
	}
	
	
	public Iterator iterator() {

		//return an iterator on which people cannot call delete(). Any modification
		//of dependent objects should go through business methods on its containing entity bean.
		return Collections.unmodifiableSet(this.workingList).iterator();
	}
	
	
	public boolean remove(Object o) {

		//if we are removing an item that was initially in the database but not modified in this transaction
		if (this.original.contains(o))
		{
			//then remove this object from our original and working lists, and add
			//it to our list of objects to delete
			this.original.remove(o);
			this.workingList.remove(o);
			this.deleted.add(o);
		}
		//if we are removing an item initially in the database and modified in this transaction
		else if (this.updatedMap.containsKey(o))
		{
			this.workingList.remove(o);

			//remove original db object from modified list and schedule it for deletion
			this.deleted.add( this.updatedMap.remove(o));
		}
		//it wasn't initially in the DB, but may be a new 
		//and thus would only be contained in our workinglist
		else if (this.workingList.contains(o))
		{
			//remove this object but don't add it to the deleted list
			//because it was never in the database
			this.workingList.remove(o);
		}
		else //this object is not part of this set
		{
			return false;
		}

		return true;
	}
	
	
	public boolean removeAll(Collection c) throws UnsupportedOperationException
	{
		throw new UnsupportedOperationException();
	}
	
	
	public boolean retainAll(Collection c) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}
	
	
	public int size() {

		//our working list reflects the current actual dependent objects in this set
		return this.workingList.size();
	}
	
	
	public java.lang.Object[] toArray() {
		return this.workingList.toArray();
	}
	
	
	public java.lang.Object[] toArray(java.lang.Object[] a) {
		return this.workingList.toArray(a);
	}
	
	
	public boolean update(Object oldObject, Object newObject) {
		
		//if oldObject was part of the original database set and not previously updated
		if (this.original.contains(oldObject))
		{   //if our object was in the db and not previously updated, it would still be in our original set

			//remove oldObject from this set
			this.original.remove(oldObject);
			this.workingList.remove(oldObject);

			//add new updated object to our working list
			this.workingList.add(newObject);
			
			//schedule oldObject to be deleted from the database
			this.updatedMap.put(newObject, oldObject);
		}
		//if oldObject is an already modified version of an object that was in the
		//database before this transaction
		else if (this.updatedMap.containsKey(oldObject))
		{
			//remove oldObject from our working list
			this.workingList.remove(oldObject);

			//add new updated object to our working list
			this.workingList.add(newObject);				
			
			//replace oldObject *key* with newObject *key* in our map of updated objects.
			//the "value" of this new "key" in the hash map is the original object
			//that was present in the db and at the beginning of this transaction
			this.updatedMap.put(newObject, this.updatedMap.get(oldObject));
			this.updatedMap.remove(oldObject);

			//add new updated object to our working list
			this.workingList.add(newObject);		
		}
		//if oldObject is "new" - created sometime during this transaction
		else if (this.workingList.contains(oldObject))
		{
			//remove it from our working list
			this.workingList.remove(oldObject);

			//add new updated object to our working list
			this.workingList.add(newObject);
			
			//don't schedule it to be deleted or updated from DB, since it was never
			//in the DB to begin with
		}
		else //old object was never in this list list at all
			return false;
			
		return true;

	}
}
