package examples.ejb20.basic.beanManagedDependentValueObjects;

import java.io.Serializable;
import java.sql.*;
import java.util.*;
import javax.ejb.*;
import javax.naming.*;
import javax.sql.DataSource;
import examples.ejb20.basic.beanManagedDependentValueObjects.*;

/**
 * Resume Bean - Shows how an entity bean with a one to one and a one to many
 * dependent object relationship can be implemented in BMP, and shows how to 
 * use the DependentValueSet class.
 * 
 * @author: Floyd Marinescu, Senior Architect, The Middleware Company.
 * The Middleware Company is an Enterprise Java Training company. Visit our
 * website at: www.middleware-company.com.
 * Copyright 2001 The Middleware Company.
 *
 * Note: The following EJB makes use of helper code from the Weblogic 6.0
 * example code distribution, such as taking in a primarykey as a parameter
 * to ejbCreate (useful for shipping examples, not for production), the 
 * "cleanup" methods, etc.
**/

public class ResumeBean implements EntityBean
{

	final static private boolean VERBOSE = true;
	private EntityContext ctx;


	private String resumeId; // also the primary Key
	private java.lang.String firstName;
	private java.lang.String lastName;

	//depedent value objects
	private AddressDependentValue address;
	private DependentValueSet jobEntries;

	//isModified flag to optimize address storing in ejbStore
	private boolean isAddressModified = false;

	/**
	 * 
	 * BUSINESS METHODS ==================================================================
	 *
    **/

	//Resume Bean data accessors              -------------------------------------------
	public String getFirstName()
	{
		return firstName;
	}

	public String getLastName()
	{
		return lastName;
	}

	public String getResumeId()
	{
		return resumeId;
	}

	void setFirstName(java.lang.String newFirstName)
	{
		firstName = newFirstName;
	}

	void setLastName(java.lang.String newLastName)
	{
		lastName = newLastName;
	}


	//Job dependent value object CRUD methods -------------------------------------------

	/**
	* Adds a job to our resume
	*/
	public void addJobEntry(JobDependentValue aJob) throws DuplicateDependentObjectException
	{
		if (! this.getJobs().add(aJob))
			throw new DuplicateDependentObjectException();
	}

	/**
	* Deletes job entry from this resume
	*/
	public void deleteJobEntry(JobDependentValue aJob) throws NoSuchDependentObjectException
	{
		if (!this.jobEntries.remove(aJob))
			throw new NoSuchDependentObjectException();

	}

	/**
	* Updates an existing job entry with new one.
	* This is the only way to "modify" our dependent object, since
	* job entries don't have an "identity", we can't simply "set" a new
	* job entry, we need to have a copy of the old one to know which one
	* we are talking about.
	*/
	public void updateJobEntry(JobDependentValue oldJob, JobDependentValue newJob) throws NoSuchDependentObjectException
	{
		if (!this.jobEntries.update(oldJob, newJob))
			throw new NoSuchDependentObjectException();
	}

	/**
	* Returns an array of job entries to the client
	*/
	public Object[] getJobEntries()
	{
		log("getJobEntries() called to return size: " + Integer.toString(getJobs().toArray().length));

		return getJobs().toArray();
	}


	/**
	* Returns an unmodifiable iterator. This cannot be called by the client
	* since the Iterator interface is not serializable.
	* @return java.util.Iterator
	*/
	public Iterator getJobEntriesIterator()
	{
		return getJobs().iterator();
	}


	/**
	* Lazy load our job entries, this is for internal use only.
	* Clients should only be passed iterators, to force any
	* CRUD operations to go through the entity bean remote interface
	*/
	public DependentValueSet getJobs()
	{

		//check to see if our address has already been loaded
		if ( this.jobEntries != null )
		{
			return this.jobEntries;
		}

		this.loadJobEntries();
		log ("getJobs called with job size:" + jobEntries.size());

		return this.jobEntries;
	}



	//Address dependent value object methods -------------------------------------------
		
	/**
	* Lazy loads our Address Dependent Object
	*/
	public AddressDependentValue getAddress()
	{

		//check to see if our address has already been loaded
		if ( this.address != null )
		{
			return address;
		}

		this.loadAddress();

		return address;

	}
		
	/**
	* Insert the method's description here.
	*/
	public void setAddress(AddressDependentValue newAddress)
	{

		this.address = newAddress;
		this.isAddressModified = true;
	}
	
		
	/**
	 * 
	 * EJB and persistence METHODS =============================================================
	 *
    **/		
		

	/**
	 * Cleanup method, borrowed from the Weblogic 6.0 sample EJB's.
	**/
	private void cleanup(Connection con, PreparedStatement ps)
	{

		try
		{
			if (ps != null) ps.close(); 
		} catch (Exception e)
		{
			log("Error closing PreparedStatement: "+e);
			throw new EJBException (e);
		}

		try
		{
			if (con != null) con.close(); 
		} catch (Exception e)
		{
			log("Error closing Connection: " + e);
			throw new EJBException (e);

		}
	}      
	private void cleanup(Connection con, PreparedStatement ps, ResultSet rs)
	{
		try
		{
			if (rs != null) rs.close(); 
		} catch (Exception e)
		{
			log("Error closing resultset: "+e);
			throw new EJBException (e);
		}

		try
		{
			if (ps != null) ps.close(); 
		} catch (Exception e)
		{
			log("Error closing PreparedStatement: "+e);
			throw new EJBException (e);
		}

		try
		{
			if (con != null) con.close(); 
		} catch (Exception e)
		{
			log("Error closing Connection: " + e);
			throw new EJBException (e);

		}
	}        

	public void ejbActivate()
	{
		log("ejbActivate (" + id() + ")");

		//set instance specific values to null
		this.address    = null;
		this.jobEntries = null;
	}    


	public String ejbCreate(String resumeId, String firstName, String lastName, AddressDependentValue anAddress ) 
	throws CreateException
	{
		log("AccountBean.ejbCreate( id = resumeId" + ", " + "firstName = " + firstName + "lastName = " + lastName +")");
		this.resumeId = resumeId;
		this.firstName = firstName;
		this.lastName  = lastName;
		this.address = anAddress;
		this.isAddressModified = false;


		Connection con = null;
		PreparedStatement ps = null;

		try {
	  		con = getConnection();

			//insert resume data
			ps = con.prepareStatement("insert into resumes (id, firstname, lastname) values (?, ?, ?)");
			ps.setString(1, this.resumeId);
			ps.setString(2, this.firstName);
			ps.setString(3, this.lastName);	  

			if (ps.executeUpdate() != 1)
			{
				String error = "JDBC did not create any row";
				log(error);
				throw new CreateException (error);
			}

			ps.close();

			//insert address
			ps = con.prepareStatement("insert into addresses (resumeid, address1, address2, city, state, country,postalCode, phoneNumber) values (?, ?, ?,?,?,?,?,?)");
			ps.setString(1, this.resumeId);
			ps.setString(2, anAddress.getAddress1());
			ps.setString(3, anAddress.getAddress2());	  
			ps.setString(4, anAddress.getCity());
			ps.setString(5, anAddress.getState());
			ps.setString(6, anAddress.getCountry());
			ps.setString(7, anAddress.getPostalCode());	  
			ps.setLong(8,    anAddress.getPhoneNumber());

			if (ps.executeUpdate() != 1)
			{
				String error = "JDBC did not create any row for address";
				log(error);
				throw new CreateException (error);
			}

			} 
		catch (SQLException sqe) 
		{
			  // Check to see if this SQLException is due to a unique constraint
			  // violation on our database table (ie. there is already a pk with the
			  // value of accountId in the table).  If so, throw a 
			  // DuplicateKeyException else throw a CreateException.
			  try 
			  {
					ejbFindByPrimaryKey(resumeId);
			  } 
			  catch(ObjectNotFoundException onfe) 
			  {
					String error = "SQLException: " + sqe;
					log(error);
					throw new CreateException (error);
			  }
			  
			  String error = "An resume already exists in the database with Primary Key " + resumeId;
			  log(error);
			  throw new DuplicateKeyException(error);
		} 
		finally 
		{
		  cleanup(con, ps);
		}

		return resumeId;	
	}                    
	
	
	public String ejbFindByPrimaryKey(String pk) throws ObjectNotFoundException
	{
		log("ejbFindByPrimaryKey (" + pk + ")");

		Connection con = null;
		PreparedStatement ps = null;

		try {
			  con = getConnection();
				ps  = con.prepareStatement("select bal from ejbAccounts where id = ?");
				ps.setString(1, pk);
				ps.executeQuery();

					ResultSet rs = ps.getResultSet();
				if (rs.next())
				{
					firstName = rs.getString("firstname");
					lastName = rs.getString("lastname");
				}
				else
				{
					String error = "ejbFindByPrimaryKey: AccountBean (" + pk + ") not found";
					log(error);
					throw new ObjectNotFoundException (error);
				}
		}
		catch (SQLException sqe) 
		{
			    log("SQLException:  " + sqe);
				throw new EJBException (sqe);
		} 
		finally 
		{
		  cleanup(con, ps);
		}

		log("ejbFindByPrimaryKey (" + pk + ") found");
			return pk;
	}    
	
	
	
	
	/**
	* Loads the EJBean from the persistent storage.
	* 
	* @exception               javax.ejb.NoSuchEntityException
	*                          if the bean is not found in the database
	* @exception               javax.ejb.EJBException
	*                          if there is a communications or systems failure
	*/
	public void ejbLoad()
	{
		log("ejbLoad: (" + id() +  ")");

		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
		resumeId = (String) ctx.getPrimaryKey();

		try
		{
			con = getConnection();
			ps  = con.prepareStatement("select * from resumes where id = ?");
			ps.setString(1, resumeId);
			ps.executeQuery();

			rs = ps.getResultSet();

			//load bean with data from result set
			if (rs.next())
			{
				firstName = rs.getString("firstname");
				lastName = rs.getString("lastname");
			}
			else
			{
				String error = "ejbLoad: AccountBean (" + resumeId + ") not found";
				log(error);
				throw new NoSuchEntityException (error);
			}
		} catch (SQLException sqe)
		{
			log("SQLException:  " + sqe);
			throw new EJBException(sqe);
		} finally
		{
			cleanup(con, ps, rs);
		}
	}          
	
	
	/**
	* This method is required by the EJB Specification
	*/
	public void ejbPassivate()
	{
		log("ejbPassivate (" + id() + ")");

	}        
	
	
	
	/**
	* Required by the EJB specification, this method is not used
	* by this example.
	*
	* @param accountID         String Account Identification
	* @param initialBalance    double Initial Balance 
	*/
	public void ejbPostCreate(String resumeId, String firstName, String lastName, AddressDependentValue anAddress)
	{
		log("ejbPostCreate (" + id() + ")");
	}    
	
	
	/**
	* Deletes the EJBean from the persistent storage. 
	* 
	* @exception               javax.ejb.NoSuchEntityException
	*                          if the bean is not found in the database
	* @exception               javax.ejb.EJBException
	*                          if there is a communications or systems failure
	*/
	public void ejbRemove()
	{
		log("ejbRemove (" + id() + ")");
		// we need to get the primary key from the context because
		// it is possible to do a remove right after a find, and
		// ejbLoad may not have been called.

		Connection con = null;
		PreparedStatement ps = null;

	try {
	  con = getConnection();
			resumeId = (String) ctx.getPrimaryKey();

			//delete this bean from the DB
			ps = con.prepareStatement("delete from resumes where id = ?");
			ps.setString(1, resumeId);
			if (!(ps.executeUpdate() > 0))
			{
				String error = "ResumeBean (" + resumeId + " not found";
				log(error);
				throw new NoSuchEntityException (error);
			}

			//delete address dependent object from DB and bean
			ps = con.prepareStatement("delete from addresses where resumeid = ?");
			ps.setString(1, resumeId);
			ps.executeUpdate();
			this.address = null;

			//delete job entry dependent objects from DB and bean
			ps = con.prepareStatement("delete from jobentries where resumeid = ?");
			ps.setString(1, resumeId);
			ps.executeUpdate();
			this.jobEntries = null;

			} 
			catch (SQLException sqe) 
			{
			  	log("SQLException:  " + sqe);
  			  	throw new EJBException (sqe);
			} 
			finally 
			{
				cleanup(con, ps);
			}
  	}        
	
	
	
	/**
	* Stores the EJBean in the persistent storage.
	* 
	* @exception               javax.ejb.NoSuchEntityException
	*                          if the bean is not found in the database
	* @exception               javax.ejb.EJBException
	*                          if there is a communications or systems failure
	*/
	public void ejbStore()
	{
		log("ejbStore (" + id() + ")");

		Connection con = null;
		PreparedStatement ps = null;

	try {
	  con = getConnection();

			//update resume settings
			ps = con.prepareStatement("update resumes set firstname = ?, lastname = ? where id = ?");
			ps.setString(1, this.firstName);
			ps.setString(2, this.lastName);
			ps.setString(3, this.resumeId);

			if (!(ps.executeUpdate() > 0))
			{
				String error = "ejbStore: ResumeBean (" + resumeId + ") not updated";
				log(error);
				throw new NoSuchEntityException (error);
			}

			//update job entries if loaded (if not loaded, nothing to change)
			if (this.jobEntries != null)
			{
				//delete jobs objects that were deleted in this transaction
				Iterator deleted = this.jobEntries.getObjectsToDelete().iterator();
				while (deleted.hasNext())
				{
					JobDependentValue aJobEntry = (JobDependentValue) deleted.next();

					ps = con.prepareStatement("delete from jobentries where resumeid = ? and company = ? and jobposition = ? and begindate = ? and enddate = ?");
					ps.setString(1, this.resumeId);
					ps.setString(2, aJobEntry.getCompany());
					ps.setString(3, aJobEntry.getPosition());
					ps.setTimestamp(4, new java.sql.Timestamp(aJobEntry.getBeginDate().getTime()));
					ps.setTimestamp(5, new java.sql.Timestamp(aJobEntry.getEndDate().getTime()));

					if (!(ps.executeUpdate() > 0))
					{
						String error = "ejbStore: Job Entry (" + aJobEntry.getCompany() + ") not deleted";
						log(error);
						throw new NoSuchEntityException (error);	
					}
				}

				//update job entries that were modified in this transaction
				Map updatedObjectsMap = this.jobEntries.getObjectsToUpdate();
				Iterator keys = updatedObjectsMap.keySet().iterator();
				while (keys.hasNext())
				{
					JobDependentValue newJobEntry = (JobDependentValue) keys.next();
					JobDependentValue oldJobEntry = (JobDependentValue) updatedObjectsMap.get(newJobEntry);

					ps = con.prepareStatement("update jobentries set company = ?, jobposition = ?, begindate = ?, enddate = ?, description = ? where resumeid = ? and company = ? and jobposition = ? and begindate = ? and enddate = ?");

					//populate statement with data to update
					ps.setString(1, newJobEntry.getCompany());
					ps.setString(2, newJobEntry.getPosition());
					ps.setTimestamp(3, new java.sql.Timestamp(newJobEntry.getBeginDate().getTime()));
					ps.setTimestamp(4, new java.sql.Timestamp(newJobEntry.getEndDate().getTime()));
					ps.setString(5, newJobEntry.getJobDescription());

					//populate statement with old data to be updated
					ps.setString(6, this.resumeId);
					ps.setString(7, oldJobEntry.getCompany());
					ps.setString(8, oldJobEntry.getPosition());
					ps.setTimestamp(9, new java.sql.Timestamp(oldJobEntry.getBeginDate().getTime()));
					ps.setTimestamp(10, new java.sql.Timestamp(oldJobEntry.getEndDate().getTime()));

					if (!(ps.executeUpdate() > 0))
					{
						String error = "ejbStore: Job Entry (" + newJobEntry.getCompany() + ") not updated";
						log(error);
						throw new EJBException(error);	
					}
				}

				//now insert any new job entries
				Iterator insert = this.jobEntries.getObjectsToInsert().iterator();
				while (insert.hasNext())
				{
					JobDependentValue aJobEntry = (JobDependentValue) insert.next();

					ps = con.prepareStatement("insert into jobentries (resumeid, company, jobposition, begindate, enddate, description) values (?,?,?,?,?,?)");
					ps.setString(1, this.resumeId);
					ps.setString(2, aJobEntry.getCompany());
					ps.setString(3, aJobEntry.getPosition());
					ps.setTimestamp(4, new java.sql.Timestamp(aJobEntry.getBeginDate().getTime()));
					ps.setTimestamp(5, new java.sql.Timestamp(aJobEntry.getEndDate().getTime()));
					ps.setString(6, aJobEntry.getJobDescription());

					if (!(ps.executeUpdate() > 0))
					{
						String error = "ejbStore: Job Entry (" + aJobEntry.getCompany() + ") not deleted";
						log(error);
						throw new EJBException (error);	
					}
				}

				//notify our job entries set that it has been synchronized with the DB
				this.jobEntries.databaseHasBeenUpdated();
			}

			//update address only if loaded
			if (this.address != null && this.isAddressModified)
			{
				ps = con.prepareStatement("update addresses set address1 = ?, address2 = ?, city = ?, state = ?, country = ?,postalCode = ?, phoneNumber = ? where resumeid = ?");
				ps.setString(1, this.address.getAddress1());
				ps.setString(2, this.address.getAddress2());	  
				ps.setString(3, this.address.getCity());
				ps.setString(4, this.address.getState());
				ps.setString(5, this.address.getCountry());
				ps.setString(6, this.address.getPostalCode());	  
				ps.setLong(7,    this.address.getPhoneNumber());
				ps.setString(8, this.resumeId);

				if (!(ps.executeUpdate() > 0))
				{
					String error = "ejbStore: address for resume (" + resumeId + ") not updated";
					log(error);
					throw new NoSuchEntityException (error);
				}
			}

			} catch(SQLException sqe) {
	  log("SQLException:  " + sqe);
			throw new EJBException (sqe);
			} finally {
	  cleanup(con, ps);
			}
  	}                                                                        

	/**
	* Gets current connection to the connection pool.
	*
	* @return                  Connection
	* @exception               javax.ejb.EJBException
	*                          if there is a communications or systems failure
	*/
	private Connection getConnection()
	throws SQLException
	{
		InitialContext initCtx = null;
	try {
	  initCtx = new InitialContext();
			DataSource ds = (javax.sql.DataSource)
				initCtx.lookup("java:comp/env/jdbc/demoPool");
			return ds.getConnection();
			} catch(NamingException ne) {
	  log("UNABLE to get a connection from demoPool!");
			log("Please make sure that you have setup the connection pool properly");
			throw new EJBException(ne);
			} finally {
	  try {
		if(initCtx != null) initCtx.close();
				} catch(NamingException ne) {
		log("Error closing context: " + ne);
				throw new EJBException(ne);
				}
	}
  	}    

	/**
	* Called at Lazy Load time to load an address dependent object
	*/
	public void loadAddress()
	{

		log("loading address for resume: (" + id() +  ")");

		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
	
	try {
	  con = getConnection();
			ps  = con.prepareStatement("select * from addresses where resumeid = ?");
			ps.setString(1, this.resumeId);
			ps.executeQuery();

			rs = ps.getResultSet();

			//load bean with data from result set
			if (rs.next())
			{
				this.address = new AddressDependentValue( rs.getString("address1"),rs.getString("address2"),rs.getString("city"),rs.getString("state"),rs.getString("country"),rs.getString("postalcode"),rs.getInt("phonenumber"));
			}
			else
			{
				String error = "Error: No address found for resume: " + resumeId;
				log(error);
				throw new NoSuchEntityException (error);
			}

			//check to make sure no more addresses exist (this is a one to one relationship)
			if (rs.next())
			{
				log("Inconsistent data: multiple addresses found for resume:" + resumeId);
			}

			} catch (SQLException sqe) {
	  log("SQLException:  " + sqe);
			throw new EJBException(sqe);
			} finally {
	  cleanup(con, ps, rs);
			}
  	}         
	/**
	* Load our collection of job entries from the database
	*/
	public void loadJobEntries()
	{

		log("loading job entries for resume: (" + id() +  ")");

		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;

	try {
	  con = getConnection();
			ps = con.prepareStatement("select * from jobentries where resumeid = ?");
			ps.setString(1, this.resumeId);
			ps.executeQuery();

			rs = ps.getResultSet();

			//create new job entries dependent object set
			this.jobEntries = new DependentValueSet();
			JobDependentValue aJob;

			//load bean with data from result set
			while (rs.next())
			{
				aJob = new JobDependentValue( rs.getString("company"),rs.getString("jobposition"),rs.getTimestamp("begindate"),rs.getTimestamp("enddate"),rs.getString("description"));
				this.jobEntries.addFromDatabase(aJob);
			}

			} catch (SQLException sqe) {
	  log("SQLException:  " + sqe);
			throw new EJBException(sqe);
			} finally {
	  cleanup(con, ps, rs);
			}
  	}     

  	// Return a String that contains this beans id
  	private String id()
  	{
  		return "PK = " + (String) ctx.getPrimaryKey();
  	}  


	private void log(String s)
	{
		if (VERBOSE) System.out.println(s);
	}  

	/**
	* Sets the EntityContext for the EJBean.
	* 
	* @param ctx               EntityContext 
	*/
	public void setEntityContext(EntityContext ctx)
	{
		log("setEntityContext called");
		this.ctx = ctx;
	}  

	/**
	* Unsets the EntityContext for the EJBean.
	*/
	public void unsetEntityContext()
	{
		log("unsetEntityContext (" + id() + ")");
		this.ctx = null;
	}  

}

