package examples.bmp;

import java.sql.*;
import javax.naming.*;
import javax.ejb.*;
import java.util.*;

/**
 * Demonstration Bean-Managed Persistent Entity Bean.
 * This Entity Bean represents a Bank Account.
 */
public class AccountBean 
    implements EntityBean {

    protected EntityContext ctx;

    //
    // Bean-managed state fields
    //

    private String accountID;	// PK
    private String ownerName;
    private double balance;

    public AccountBean() {
        System.out.println("New Bank Account Entity Bean Java Object created by EJB Container.");
    }

    //
    // Business Logic Methods
    //

    /**
     * Deposits amt into account.
     */
    public void deposit(double amt) 
        throws AccountException 
    {
        System.out.println("deposit(" + amt + ") called.");

        balance += amt;
    }

    /**
     * Withdraws amt from bank account.
     * @throw AccountException thrown if amt > available balance
     */
    public void withdraw(double amt) 
        throws AccountException 
    {
        System.out.println("withdraw(" + amt + ") called.");

        if (amt > balance) {
            throw new AccountException("Your balance is " + balance + 
                                       "!  You cannot withdraw " + amt + "!");
        }

        balance -= amt;
    }

    // Getter/setter methods on Entity Bean fields

    public double getBalance() {
        System.out.println("getBalance() called.");
        return balance;
    }

    public void setOwnerName(String name) {
        System.out.println("setOwnerName() called.");
        ownerName = name;
    }

    public String getOwnerName() {
        System.out.println("getOwnerName() called.");
        return ownerName;
    }

    public String getAccountID() {
        System.out.println("getAccountID() called.");
        return accountID;
    }

    public void setAccountID(String id) {
        System.out.println("setAccountID() called.");
        this.accountID = id;
    }

    /**
     * This home business method is independent of any
     * particular account instance.  It returns the total
     * of all the bank accounts in the bank.
     */
    public double ejbHomeGetTotalBankValue() throws AccountException {
        PreparedStatement pstmt = null;
        Connection conn = null;

        try {
            System.out.println("ejbHomeGetTotalBankValue()");

            /*
             * Acquire DB connection
             */
            conn = getConnection();

            /*
             * Get the total of all accounts
             */
            pstmt = conn.prepareStatement("select sum(balance) as total from accounts");
            ResultSet rs = pstmt.executeQuery();

            /*
             * Return the sum
             */
            if (rs.next()) {
                return rs.getDouble("total");
            }
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new AccountException(e);
        }
        finally {
            /*
             * Release DB Connection for other beans
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }

        throw new AccountException("Error!");
    }

    //
    // EJB-required methods
    //

    /**
     * Called by Container.  Implementation can acquire
     * needed resources.
     */
    public void ejbActivate() {
        System.out.println("ejbActivate() called.");
    }

    /**
     * Removes entity bean data from the database.
     * Corresponds to when client calls home.remove().
     */
    public void ejbRemove() throws RemoveException {
        System.out.println("ejbRemove() called.");

        /*
         * Remember that an entity bean class can be used to
         * represent different data instances.  So how does
         * this method know which instance in the database
         * to delete?
         *
         * The answer is to query the container by calling
         * the entity context object.  By retrieving the
         * primary key from the entity context, we know
         * which data instance, keyed by the PK, that we
         * should delete from the DB.
         */
        AccountPK pk = (AccountPK) ctx.getPrimaryKey();
        String id = pk.accountID;

        PreparedStatement pstmt = null;
        Connection conn = null;
        try {
            /*
             * 1) Acquire a new JDBC Connection
             */
            conn = getConnection();

            /*
             * 2) Remove account from the DB
             */
            pstmt = conn.prepareStatement("delete from accounts where id = ?");
            pstmt.setString(1, id);

            /*
             * 3) Throw a system-level exception if something
             * bad happened.
             */
            if (pstmt.executeUpdate() == 0) {
                throw new RemoveException("Account " + pk + " failed to be removed from the database");
            }
        }
        catch (Exception ex) {
            throw new EJBException(ex.toString());
        }
        finally {
            /*
             * 4) Release the DB Connection
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }
    }

    /**
     * Called by Container.  Releases held resources for
     * passivation.
     */
    public void ejbPassivate() {
        System.out.println("ejbPassivate () called.");
    }

    /**
     * Called by the container.  Updates the in-memory entity
     * bean object to reflect the current value stored in
     * the database.
     */
    public void ejbLoad() {
        System.out.println("ejbLoad() called.");

        /*
         * Again, query the Entity Context to get the current
         * Primary Key, so we know which instance to load.
         */
        AccountPK pk = (AccountPK) ctx.getPrimaryKey();
        String id = pk.accountID;

        PreparedStatement pstmt = null;
        Connection conn = null;
        try {
            /*
             * 1) Acquire a new DB Connection
             */
            conn = getConnection();

            /*
             * 2) Get account from the DB, querying
             *    by account ID
             */
            pstmt = conn.prepareStatement("select ownerName, balance from accounts where id = ?");
            pstmt.setString(1, id);
            ResultSet rs = pstmt.executeQuery();
            rs.next();
            ownerName = rs.getString("ownerName");
            balance = rs.getDouble("balance");
        }
        catch (Exception ex) {
            throw new EJBException("Account " + pk + " failed to load from database", ex);
        }
        finally {
            /*
             * 3) Release the DB Connection
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }

    }

    /**
     * Called from the Container.  Updates the database
     * to reflect the current values of this in-memory
     * entity bean instance.
     */
    public void ejbStore() {
        System.out.println("ejbStore() called.");

        PreparedStatement pstmt = null;
        Connection conn = null;
        try {
            /*
             * 1) Acquire a new DB Connection
             */
            conn = getConnection();

            /*
             * 2) Store account in DB
             */
            pstmt = conn.prepareStatement("update accounts set ownerName = ?, balance = ? where id = ?");
            pstmt.setString(1, ownerName);
            pstmt.setDouble(2, balance);
            pstmt.setString(3, accountID);
            pstmt.executeUpdate();

        }
        catch (Exception ex) {
            throw new EJBException("Account " + accountID + " failed to save to database", ex);
        }
        finally {
            /*
             * 3) Release the DB Connection
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }
    }

    /**
     * Called by the container.  Associates this bean
     * instance with a particular context.  We can query
     * the bean properties which customize the bean here.
     */
    public void setEntityContext(EntityContext ctx) {
        System.out.println("setEntityContext called");
        this.ctx = ctx;
    }

    /**
     * Called by Container.  Disassociates this bean
     * instance with a particular context environment.
     */
    public void unsetEntityContext() {
        System.out.println("unsetEntityContext called");
        this.ctx = null; 
    }

    /**
     * Called after ejbCreate().  Now, the Bean can retrieve
     * its EJBObject from its context, and pass it as
     * a 'this' argument.
     */
    public void ejbPostCreate(String accountID, String ownerName) {
    }

    /**
     * This is the initialization method that corresponds to the
     * create() method in the Home Interface.
     *
     * When the client calls the Home Object's create() method,
     * the Home Object then calls this ejbCreate() method.
     *
     * @return The primary key for this account
     */
    public AccountPK ejbCreate(String accountID, String ownerName) throws CreateException {

        PreparedStatement pstmt = null;
        Connection conn = null;
        try {
            System.out.println("ejbCreate() called.");
            this.accountID = accountID;
            this.ownerName = ownerName;
            this.balance = 0;

            /*
             * Acquire DB connection
             */
            conn = getConnection();

            /*
             * Insert the account into the database
             */
            pstmt = conn.prepareStatement("insert into accounts (id, ownerName, balance) values (?, ?, ?)");
            pstmt.setString(1, accountID);
            pstmt.setString(2, ownerName);
            pstmt.setDouble(3, balance);
            pstmt.executeUpdate();

            /*
             * Generate the Primary Key and return it
             */
            return new AccountPK(accountID);
        }
        catch (Exception e) {
            throw new CreateException(e.toString());
        }
        finally {
            /*
             * Release DB Connection for other beans
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }
    }

    /**
     * Finds a Account by its primary Key
     */
    public AccountPK ejbFindByPrimaryKey(AccountPK key) throws FinderException {
        PreparedStatement pstmt = null;
        Connection conn = null;
        try {
            System.out.println("ejbFindByPrimaryKey(" + key + ") called");

            /*
             * Acquire DB connection
             */
            conn = getConnection();

            /*
             * Find the Entity in the DB
             */
            pstmt = conn.prepareStatement("select id from accounts where id = ?");
            pstmt.setString(1, key.toString());
            ResultSet rs = pstmt.executeQuery();
            rs.next();

            /*
             * No errors occurred, so return the Primary Key
             */
            return key;
        }
        catch (Exception e) {
            throw new FinderException(e.toString());
        }
        finally {
            /*
             * Release DB Connection for other beans
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }
    }

    /**
     * Finds all Accounts by a name
     */
    public Collection ejbFindByOwnerName(String name) throws FinderException {
        PreparedStatement pstmt = null;
        Connection conn = null;
        Vector v = new Vector();

        try {
            System.out.println("ejbFindByOwnerName(" + name + ") called");

            /*
             * Acquire DB connection
             */
            conn = getConnection();

            /*
             * Find the primary keys in the DB
             */
            pstmt = conn.prepareStatement("select id from accounts where ownerName = ?");
            pstmt.setString(1, name);
            ResultSet rs = pstmt.executeQuery();

            /*
             * Insert every primary key found into a vector
             */
            while (rs.next()) {
                String id = rs.getString("id");
                v.addElement(new AccountPK(id));
            }

            /*
             * Return the vector of primary keys
             */
            return v;
        }
        catch (Exception e) {
            throw new FinderException(e.toString());
        }
        finally {
            /*
             * Release DB Connection for other beans
             */
            try { if (pstmt != null) pstmt.close(); }
            catch (Exception e) {}
            try { if (conn != null) conn.close(); }
            catch (Exception e) {}
        }
    }

    /**
     * Gets JDBC connection from the connection pool.
     *
     * @return The JDBC connection
     */
    public Connection getConnection() throws Exception {
        try {
            Context ctx = new InitialContext();
            javax.sql.DataSource ds = (javax.sql.DataSource) ctx.lookup("java:comp/env/jdbc/bmp-account");
            return ds.getConnection();
        }
        catch (Exception e) {
            System.err.println("Could not locate datasource!  Reason:");
            e.printStackTrace();
            throw e;
        }
    }
}
