package examples.security;

import java.util.*;
import java.io.IOException;

import javax.security.auth.*;
import javax.security.auth.callback.*;
import javax.security.auth.login.*;
import javax.security.auth.spi.*;

import com.sun.enterprise.security.auth.login.PasswordCredential;

/**
 * A login module that performs password authentication.
 *
 * The purpose of this class is to actually use a callback handler
 * to collect authentication information and add it to the subject
 */

public class PasswordLoginModule 
    implements LoginModule 
{
    private Subject subject;
    /** the callback handler is the mechanism to collect authentication data */
    private javax.security.auth.callback.CallbackHandler callbackHandler;

    /** credentials: username and password */
    private String username;
    private char[] password;

    /**
     * Initializes us with a particular subject to which we will later
     * add the collected password data.
     */

    public void initialize(Subject subject, 
                           javax.security.auth.callback.CallbackHandler callbackHandler, 
                           Map sharedState, 
                           Map options) 
    {
        this.subject = subject;
        this.callbackHandler = callbackHandler;
    }

    /**
     * Authenticate the user by prompting for a username and password.
     * It is called when the client tries to login in.
     *
     * @return    true in all cases since this <code>LoginModule</code> 
     *            should not be ignored.
     * @exception FailedLoginException if the authentication fails. 
     * @exception LoginException if this <code>LoginModule</code> is unable to 
     *            perform the authentication.
     */

    public boolean login() 
        throws LoginException 
    {
        // prompt for a username and password
        if (callbackHandler == null)
        {
            throw new LoginException("Error: No CallbackHandler available to collect authentication information");
        }

        // set up a name callback and a password callback
        Callback[] callbacks = new Callback[2];
        callbacks[0] = new NameCallback("username: ");
        callbacks[1] = new PasswordCallback("password: ", false);
        
        try 
        {
            // let handler handle these
            callbackHandler.handle(callbacks);

            // get authentication data 
            username = ((NameCallback)callbacks[0]).getName();
            if(username == null)
            {
                throw new LoginException("No user specified");
            }
            
            char[] tmpPassword = ((PasswordCallback)callbacks[1]).getPassword();
            
            if (tmpPassword == null) 
            {
                // treat null password as an empty password
                tmpPassword = new char[0];
            }
            password = new char[tmpPassword.length];
            System.arraycopy(tmpPassword, 0, password, 0, tmpPassword.length);
            ((PasswordCallback)callbacks[1]).clearPassword();
        } 
        catch (java.io.IOException ioe) 
        {
            throw new LoginException(ioe.toString());
        } 
        catch (UnsupportedCallbackException uce) 
        {
            throw new LoginException("Error: No Callback available to collect authentication data :"  + 
                                     uce.getCallback().toString());
        }
        catch( Exception e )
        {
            e.printStackTrace();
        }
        
        // The client side login module will always succeed. The
        // actual login will take place on the server side when the
        // security context is passed.
        return true;
    }

    /**
     * This method is called if the overall authentication succeeds
     * after potentially many login modules had their way. In our
     * simple case, we always succeed. The important part here is
     * adding the newly authenticated principal to the security
     * context.
     *
     * @return true if this method executes properly
     */

    public boolean commit() 
        throws LoginException 
    {
        // add the user name and password as credentials to the
        // security context, i.e., the Subject
        PasswordCredential pc = 
            new PasswordCredential(username,
                                   new String(password), 
                                   "fileRealm");
        
        subject.getPrivateCredentials().add(pc);
        
        username = null;
        password = null;
        return true;
    }
    
    /**
     * This method is called if the overall authentication failed
     * (even if this particular login module succeeded).  This cannot
     * happen int our simple examples.
     *
     * @return true if this method executes properly
     */
    
    public boolean abort() 
        throws LoginException 
    {
        return true;
    }

    /**
     * Logout the user and clean up.
     *
     * @return true if this method executes properly
     */
    public boolean logout() 
        throws LoginException 
    {
        username = null;
        password = null;
        return true;
    }
}
