package examples;

import java.util.*;
import java.io.*;
import java.rmi.*;

import javax.naming.*;
import javax.ejb.*;

/**
 * This is a container-managed persistent entity bean that
 * represents an order placed for goods.
 *
 * Note: This Bean could easily be extended to include other
 * things, such as:
 * - Shipping charges
 * - Shipping address vs. Billing address
 * - A date which this order is scheduled to be completed/shipped.
 */
public abstract class OrderBean 
    implements EntityBean 
{
    private static final String customerHomeName = "java:comp/env/ejb/CustomerHome";

    protected EntityContext ctx;

    //-------------------------------------------------------
    // Begin business methods
    //-------------------------------------------------------

    /**
     * This order's identification number.  It's also our
     * Primary Key. 
     * @return order id
     */
    public abstract String getOrderID();
    
    /**
     * Sets the order id of an order
     */
    public abstract void setOrderID(String id);
	
    /**
     * Returns set of the order line items EJB objects
     */
    public abstract Collection getLineItems();
    
    /**
     * Sets set of the order line items EJB objects
     */
    public abstract void setLineItems(Collection lineItems);
	
    /**
     * @returns the Customer (EJB object reference) who placed
     * this Order.
     */
    public abstract Customer getCustomer();
    
    /**
     * set customer (EJB object reference) who placed
     * this Order.
     */
    public abstract void setCustomer(Customer customer);
	
    /**
     * Returns the date this order was placed.
     */
    public abstract java.sql.Timestamp getOrderDate();
    
    /**
     * Sets the date this order
     */
    public abstract void setOrderDate(java.sql.Timestamp timestamp);
	
    /**
     * Returns status information about the order.  Could be
     * "submitted", "approved", "shipping", "delivered",
     * or "returned".
     */
    public abstract String getStatus();
    
    /**
     * Sets status information about the order.  Could be
     * "submitted", "approved", "shipping", "delivered",
     * or "returned".
     */
    public abstract void setStatus(String status);
    
    /**
     * Sets the subtotal of this order
     */

    public abstract double getSubTotal();
   
    /**
     * Returns the subtotal of this order
     */
    public abstract void setSubTotal(double subTotal);
     
    /**
     * Returns taxes of this order
     */
    public abstract double getTaxes();
   
    /**
     * Sets the taxes of this order
     */
    public abstract void setTaxes(double taxes);
	
    /**
     * Returns the price of the entire order
     */
    public double getTotalPrice() 
    {
        double totalPrice = 0;
        Iterator i = getLineItems().iterator();
        while (i.hasNext()) 
        {
            OrderLineItem item = (OrderLineItem) i.next();
            totalPrice += item.getProduct().getBasePrice();
        }
        return totalPrice;
    }

    //-------------------------------------------------------
    // End business methods
    //-------------------------------------------------------

    //-------------------------------------------------------
    // Begin EJB-Required methods.
    //-------------------------------------------------------

    /**
     * Associates this Bean instance with a particular context.
     * Once done, we can query the Context for environment
     * info, such as Bean customizations via properties.
     */
    public void setEntityContext(EntityContext ctx) 
    {
        System.out.println("Order.setEntityContext called");
        this.ctx = ctx;
    }

    /**
     * Disassociates this Bean instance with a particular
     * context environment.
     */
    public void unsetEntityContext() {
        System.out.println("Order.unsetEntityContext called");
        this.ctx = null; 
    }

    /**
     * Called directly after activating this bean instance.
     * You should acquire needed resources in this method.
     */
    public void ejbActivate() {
        System.out.println("Order.ejbActivate() called.");
    }

    /**
     * Called directly before passivating this bean instance.
     * Release any resources you acquired in ejbActivate() in
     * this method.
     */
    public void ejbPassivate() {
        System.out.println("Order.ejbPassivate () called.");
    }

    /**
     * Updates the database to reflect the current values of
     * this object.
     *
     * Since we're using Container-Managed Persistence, the
     * container-generated subclass will automatically save
     * our state.  We then do pre-processing here.
     */
    public void ejbStore() {
        System.out.println("Order.ejbStore() called.");
    }

    /**
     * Updates this object to reflect any changes to the
     * database data.
     *
     * Since we're using Container-Managed Persistence, the 
     * container-generated subclass will automatically load
     * our state.  We then do post-processing here.
     */
    public void ejbLoad() {
        System.out.println("Order.ejbLoad() called.");
    }

    /**
     * Called when new database data is created.
     *
     * When the client calls the Home Object's create() method,
     * the Home Object then calls this ejbCreate() method.
     *
     * We need to initialize our Bean's state with the parameters
     * passed from the client by calling our own abstract set
     * methods.  The Container can then inspect our Bean and
     * INSERT the corresponding database entries.
     */
    public String ejbCreate(String orderID, 
                            Customer customer, 
                            Collection lineItems) 
        throws CreateException 
    {
        System.out.println("Order.ejbCreate(" + orderID + ") called");

        setOrderID(orderID);  
        setOrderDate(new java.sql.Timestamp(System.currentTimeMillis()));
        setStatus("submitted");

        /*
         * CMP entity beans' ejbCreate() always return null.
         * The ejbCreate() method has a non-void return signature so
         * that the ejbCreate() signature matches that of a
         * BMP entity bean, allowing you to create a BMP entity bean
         * that subclasses a CMP entity bean.
         */
        return null;
    }
        
    /**
     * Called when new database data is created.
     *
     * When the client calls the Home Object's create() method,
     * the Home Object then calls this ejbCreate() method.
     *
     * We need to initialize our Bean's state with the parameters
     * passed from the client by calling our own abstract set
     * methods.  The Container can then inspect our Bean and
     * INSERT the corresponding database entries.
     */

    public String ejbCreate(String orderID, 
                            Customer customer,
                            String status,
                            double subTotal,
                            double taxes) 
        throws CreateException 
    {
        System.out.println("Order.ejbCreate(" + orderID + ") called");
        setOrderID(orderID);
        setOrderDate(new java.sql.Timestamp(System.currentTimeMillis()));
        setStatus(status);
        setSubTotal(subTotal);
        setTaxes(taxes);	

        /*
         * CMP entity beans' ejbCreate() always return null.
         * The ejbCreate() method has a non-void return signature so
         * that the ejbCreate() signature matches that of a
         * BMP entity bean, allowing you to create a BMP entity bean
         * that subclasses a CMP entity bean.
         */
        return null;
    }
    
    /**
     * This ejbCreate() accepts a String as a customer, and looksup the customer
     * on behalf of the client
     */
    public String ejbCreate(String orderID, String customerID, Collection lineItems) 
        throws CreateException 
    {
        System.out.println("Order.ejbCreate(" + orderID + ") called");

        setOrderID(orderID);  
        setOrderDate(new java.sql.Timestamp(System.currentTimeMillis()));
        setStatus("submitted");

        return null;
//         try 
//         {
//             Context ctx = new InitialContext();
//             CustomerHome home = 
//                 (CustomerHome)javax.rmi.PortableRemoteObject.narrow(
//                         ctx.lookup(customerHomeName), CustomerHome.class);                
//             Customer c = home.findByPrimaryKey(customerID);
//             return this.ejbCreate(orderID, c, lineItems);
//         }
//         catch (Exception e) 
//         {
//             e.printStackTrace();
//             throw new EJBException(e);
//         }
    }
    
    /**
     * This ejbCreate() accepts a String as a customer, and looksup the customer
     * on behalf of the client
     */
    public String ejbCreate(String orderID, 
                            String customerID,
                            String status, 
                            double subTotal,
                            double taxes) 
        throws CreateException 
    {
        setOrderID(orderID);
        setOrderDate(new java.sql.Timestamp(System.currentTimeMillis()));
        setStatus(status);
        setSubTotal(subTotal);
        setTaxes(taxes);
        return null;
	
//         try 
//         {
//             Context ctx = new InitialContext();
//             CustomerHome home = 
//                 (CustomerHome)javax.rmi.PortableRemoteObject.narrow(
//                         ctx.lookup(customerHomeName), CustomerHome.class);
//             Customer c = home.findByPrimaryKey(customerID);
//             return this.ejbCreate(orderID, c, status, subTotal,taxes);
//         }
//         catch (Exception e)
//         {
//             e.printStackTrace();
//             throw new EJBException(e);
//         }
    }

    /**
     * The Container calls this after ejbCreate().  At this
     * point in time, the bean instance is associated with its
     * own EJB Object.  You can get a reference to that
     * EJB Object by querying the context.  You'd use that
     * EJB Object reference when calling  external code,
     * and you'd like to pass a reference to yourself.
     * Sets the customer cmr-field
     */
    public void ejbPostCreate( String orderLineItemID, Customer customer, Collection lineItems) 
        throws CreateException 
    {
        System.out.println("Order.ejbPostCreate() 1  called");
        setCustomer(customer);           
    }
	
    /**
     * The Container calls this after ejbCreate().  At this
     * point in time, the bean instance is associated with its
     * own EJB Object.  You can get a reference to that
     * EJB Object by querying the context.  You'd use that
     * EJB Object reference when calling  external code,
     * and you'd like to pass a reference to yourself.
     * Sets the customer cmr-field
     */
    public void ejbPostCreate(String orderLineItemID, String customerID, Collection lineItems) 
        throws CreateException 
    {
        System.out.println("Order.ejbPostCreate() called");
        try
        {
            Context ctx = new InitialContext();
            CustomerHome home = 
                (CustomerHome)javax.rmi.PortableRemoteObject.narrow(
                    ctx.lookup(customerHomeName), CustomerHome.class);
            Customer customer = home.findByPrimaryKey(customerID);
            setCustomer(customer);
        }
        catch (Exception e) 
        {
            e.printStackTrace();
            throw new EJBException(e);
        }   
    }
    
    /**
     * The Container calls this after ejbCreate().  At this
     * point in time, the bean instance is associated with its
     * own EJB Object.  You can get a reference to that
     * EJB Object by querying the context.  You'd use that
     * EJB Object reference when calling  external code,
     * and you'd like to pass a reference to yourself.
     * Sets the customer cmr-field
     */
    public void ejbPostCreate(String orderLineItemID, 
                              Customer customer,
                              String status,
                              double subTotal,
                              double taxes) 
        throws CreateException 
    {
        System.out.println("Order.ejbPostCreate() 1  called");
        setCustomer(customer);                
    }
     
    /**
     * The Container calls this after ejbCreate().  At this
     * point in time, the bean instance is associated with its
     * own EJB Object.  You can get a reference to that
     * EJB Object by querying the context.  You'd use that
     * EJB Object reference when calling  external code,
     * and you'd like to pass a reference to yourself.
     * Sets the customer cmr-field
     */
    public void ejbPostCreate(String orderLineItemID, 
                              String customerID,
                              String status, 
                              double subTotal,
                              double taxes) 
        throws CreateException 
    {
        System.out.println("Order.ejbPostCreate() called");
        try
        {
            Context ctx = new InitialContext();
            CustomerHome home = 
                (CustomerHome)javax.rmi.PortableRemoteObject.narrow(
                     ctx.lookup(customerHomeName), CustomerHome.class);
            Customer customer = home.findByPrimaryKey(customerID);
            setCustomer(customer);
        }
        catch (Exception e) 
        {
            e.printStackTrace();
            throw new EJBException(e);
        }
    }

    /**
     * Called before the container removes entity bean data
     * from the database.  Corresponds to when client calls
     * home.remove().
     */
    public void ejbRemove() 
        throws RemoveException 
    {
        System.out.println("Order.ejbRemove() called.");
    }

    // No finder methods - they are implemented by Container

    //-------------------------------------------------------
    // End EJB required methods
    //------------------------------------------------------
}
